using System;
using GovernmentGateway;
using MSIBPlusPack.CommerceServer.Runtime;

namespace GovernmentGatewayAuthenticationPlugin 
{
	/// <summary>
	/// Allows a Commerce Server user profile to be associated with a Government Gateway user. The Government Gateway token is stored, if configured by the module plugin.
	/// </summary>
	public class GGAuthPlugin :IExternalAuthenticationPlugIn
	{
		#region IExternalAuthenticationPlugIn Members

		/// <summary>
		/// Authenticates a user against the government gateway. 
		/// </summary>
		/// <param name="username">Username to use against government gateway</param>
		/// <param name="password">Password to use against Government Gateway.</param>
		/// <param name="ticket">returned ticket if successfully logged in</param>
		/// <param name="ticketExpirationDate">Date and time the ticket expires</param>
		/// <returns>true if authentication was successful, false otherwise</returns>
		public bool AuthenticateUser(string username, string password, ref string ticket, ref DateTime ticketExpirationDate)
		{

			GovernmentGateway.Security.GatewayToken token = GovernmentGateway.Security.GatewaySecurityAgent.GetGatewayToken(username, GovernmentGateway.Security.GatewaySecurityAgent.PreparePassword(password), GovernmentGateway.Security.GatewayTokenUse.Standard);
			if(token != null)
			{
				if(token.IsExpiredPassword)
					throw new System.Security.SecurityException("Government Gateway Password has expired");
				ticket = token.Xml;
				ticketExpirationDate = token.Expires;
				return true;
			}

			// Execute Method
			//GovernmentGateway.RegistrationAndEnrolment.AgentPortal portal = new GovernmentGateway.RegistrationAndEnrolment.AgentPortal(token);
			//string returnXml = portal.GsoGetUserDetails("<GsoGetUserDetailsXmlInput xmlns=\"urn:GSO-System-Services:external:1.65:GsoGetUserDetailsXmlInput\"/>");
			return false;
		}

		/// <summary>
		/// Not required. Government Gateway is configured at the web service level and so is not required to be passed through its configuration. 
		/// </summary>
		/// <param name="plugInConfig">Configuration for the plugin</param>
		public void Init(System.Collections.Specialized.NameValueCollection plugInConfig)
		{
		}

		/// <summary>
		/// Renewing tickets is not permitted for the Government Gateway. Use ValidateTicket to check and if necessary prompt the user to reauthenticate.
		/// </summary>
		/// <param name="ticket">not required</param>
		/// <param name="ticketExpirationDate">not required</param>
		/// <returns>false under all circumstances</returns>
		public bool RenewTicket(ref string ticket, ref DateTime ticketExpirationDate)
		{
			return false;
		}

		/// <summary>
		/// Validates that the current Government Gateway token is still valid. 
		/// </summary>
		/// <param name="ticket">Government Gateway token in a serialized XML document.</param>
		/// <returns>true if the token is valid and current, false otherwise</returns>
		public bool ValidateTicket(ref string ticket)
		{
			GovernmentGateway.Security.GatewayToken token = new GovernmentGateway.Security.GatewayToken(ticket);
			return token.IsCurrent;			
		}

		/// <summary>
		/// Not required - no resources are managed that may need to be disposed. 
		/// </summary>
		/// <param name="disposing"></param>
		public void Dispose(bool disposing)
		{
		}

		#endregion
	}
}
